<?php

namespace App\Filters;

use CodeIgniter\Filters\FilterInterface;
use CodeIgniter\HTTP\RequestInterface;
use CodeIgniter\HTTP\ResponseInterface;
use App\Helpers\JwtHelper;
use App\Models\UsuarioModel;

class AuthFilter implements FilterInterface
{
    public function before(RequestInterface $request, $arguments = null)
    {
        $authHeader = $request->getHeaderLine('Authorization');
        
        if (empty($authHeader)) {
            log_message('error', 'AuthFilter - No se encontró header Authorization');
            return service('response')
                ->setJSON(['success' => false, 'message' => 'Token no proporcionado'])
                ->setStatusCode(401);
        }

        // Extraer el token (remover "Bearer ")
        if (!preg_match('/Bearer\s(\S+)/', $authHeader, $matches)) {
            log_message('error', 'AuthFilter - Formato de token incorrecto');
            return service('response')
                ->setJSON(['success' => false, 'message' => 'Formato de token incorrecto'])
                ->setStatusCode(401);
        }
        
        $token = $matches[1];
        log_message('debug', 'AuthFilter - Token extraído: ' . $token);

        // Decodificar y validar el token JWT
        $decodedToken = JwtHelper::decodeToken($token);

        if ($decodedToken === null) {
            log_message('error', 'AuthFilter - Token inválido o expirado: ' . $token);
            return service('response')
                ->setJSON(['success' => false, 'message' => 'Token inválido o expirado'])
                ->setStatusCode(401);
        }

        // El token es válido, buscar al usuario en la base de datos
        $model = new UsuarioModel();
        $usuario = $model->find($decodedToken->data->id_usuario);

        // Verificar si el usuario existe y está activo
        if ($usuario === null || $usuario['estado'] !== 'activo') {
            log_message('error', 'AuthFilter - Usuario no encontrado o inactivo para el token: ' . $token);
            return service('response')
                ->setJSON(['success' => false, 'message' => 'Usuario no autorizado'])
                ->setStatusCode(401);
        }

        log_message('debug', 'AuthFilter - Usuario autenticado: ' . $usuario['username']);

        // Guardar el usuario en el request para usarlo en los controladores
        $request->usuario = $usuario;
        
        return $request;
    }

    public function after(RequestInterface $request, ResponseInterface $response, $arguments = null)
    {
        // No hacer nada después
    }
}
